<?php
use Codexpert\CoDesigner_Pro\Helper;

if( ! function_exists( 'get_plugin_data' ) ) {
	require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
}

/**
 * Gets the site's base URL
 * 
 * @uses get_bloginfo()
 * 
 * @return string $url the site URL
 */
if( ! function_exists( 'codesigner_pro_site_url' ) ) :
function codesigner_pro_site_url() {
	$url = get_bloginfo( 'url' );

	return $url;
}
endif;

/**
 * Get Archive Template Id
 *
 * @since 3.0
 * @author Jakaria Istauk <jakariamd35@gmail.com>
 */

if( !function_exists( 'wcd_get_archive_template_id' ) ) :
function wcd_get_archive_template_id( $template_type ){

    $object = get_queried_object();

    if ( is_null( $object ) && ( !is_home() || !is_front_page() ) ) return;
    
    if ( is_home() || is_front_page() ) {
        $id     = 'home';
        $screen = 'page';
    }
    elseif ( function_exists( 'is_shop' ) && is_shop() ) {
        $id     = wc_get_page_id( 'shop' );
        $screen = 'page';
    }
    elseif ( is_tax() ) {
        $id     = $object->term_id;
        $screen = 'tax';
    }
    elseif ( is_singular() ) {
        $screen = $object->post_type;
        $id     = $object->ID;
    }
    else{
        $id     = $object->ID;
        $screen = $object->post_type;
    }


    $args = [  
        'post_type'         => 'elementor_library',
        'post_status'       => 'publish',
        'posts_per_page'    => 1, 
        'orderby'           => 'modified',
        'order'             => 'DESC',
        'meta_query'        => [
            'relation'      => 'AND',
            [
                'key'       => '_elementor_template_type',
                'value'     => $template_type,
            ],
            [
                'relation'  => 'or',
                [
                    'key'       => "wl_{$screen}_includes",
                    'value'     => '"' . $id . '"',
                    'compare'   => 'LIKE'
                ],
                [
                    'key'       => "wl_{$screen}_includes",
                    'value'     => '"0"',
                    'compare'   => 'LIKE'
                ],
            ]
        ]
    ];
    
    $result     = new \WP_Query( $args ); 
    $template   = $result->post;

    if ( empty( $template ) ) return;

    return $template->ID;
}
endif;

/**
 * Return the template types
 *
 * @since 3.0
 * @author Jakaria Istauk <jakariamd35@gmail.com>
 */
if( !function_exists( 'wcd_get_template_types' ) ) :
function wcd_get_template_types(){

    $types['wl-header'] = [
        'class' => 'Header',
        'label' => __( 'CoDesigner Header', 'codesigner-pro' )
    ];
    $types['wl-footer'] = [
        'class' => 'Footer',
        'label' => __( 'CoDesigner Footer', 'codesigner-pro' )
    ];
    $types['wl-archive'] = [
        'class' => 'Archive',
        'label' => __( 'CoDesigner Product Archive', 'codesigner-pro' )
    ];
    $types['wl-single'] = [
        'class' => 'Single',
        'label' => __( 'CoDesigner Single Product', 'codesigner-pro' )
    ];
    $types['wl-tab'] = [
        'class' => 'Tab',
        'label' => __( 'CoDesigner Tab', 'codesigner-pro' )
    ];
    $types['wl-email'] = [
        'class' => 'Email',
        'label' => __( 'CoDesigner Email', 'codesigner-pro' )
    ];

    return apply_filters( 'codesigner-pro_template_types', $types );
}
endif;

/**
 * List of conditions
 *
 * @since 1.0
 *
 * @return []
 */
if( !function_exists( 'wcd_show_content_to' ) ) :
function wcd_show_content_to() {

    $rules = [
        'user'          => __( 'User Status', 'codesigner-pro' ),
        'user_id'       => __( 'User ID', 'codesigner-pro' ),
        'user_role'     => __( 'User Role', 'codesigner-pro' ),
        'date_time'     => __( 'Date Time', 'codesigner-pro' ),
        'query_string'  => __( 'Query String', 'codesigner-pro' ),
        'customer_purchased' => __( 'Customer\'s Product', 'codesigner-pro' ),
        'customer_order' => __( 'Customer\'s Order', 'codesigner-pro' ),
    ];

    return apply_filters( 'wcd_show_content_to', $rules );
}
endif;

/**
 * List of user roles
 *
 * @since 1.0
 *
 * @return []
 */
if( !function_exists( 'wcd_show_content_to_roles' ) ) :
function wcd_show_content_to_roles() {

    global $wp_roles;

    if( empty( $wp_roles ) ) return [];
    $all_roles = $wp_roles->roles;
    $roles = [];

    foreach ( $all_roles as $key => $role ) {
        $roles[ $key ] = $role['name'];
    }

    return $roles;

    return apply_filters( 'wcd_user_roles_to_show_content', $roles );
}
endif;

/**
 * Return elementor template library list
 * 
 * @param string $template_type ex: 'wl-tab'
 *  
 */
if( !function_exists( 'wcd_get_template_list' ) ) :
function wcd_get_template_list( $template_type = 'wl-tab' ){

    $args = [  
        'post_type'      => 'elementor_library',
        'post_status'    => 'publish',
        'posts_per_page' => -1, 
        'order'          => 'DESC',
        'meta_query'     => [
            'relation'   => 'AND',
            [
                'key'       => '_elementor_template_type',
                'value'     => $template_type,
            ]
        ]
    ];

    $result = new \WP_Query( $args ); 
    $_tabs  = $result->posts;

    $tabs = [];
    foreach ( $_tabs as $tab ) {
        $tabs[ $tab->ID ] = $tab->post_title;
    }   

    return $tabs;
}
endif;

/**
 * texonomy formatted for filter
 *
 * @since 1.0
 *
 * @return []
 */
if( !function_exists( 'wcd_formated_taxonomy' ) ) :
function wcd_formated_taxonomy() {
    $_taxonomies = get_object_taxonomies( 'product' );
    $taxonomies = [];
    foreach ( $_taxonomies as $_taxonomy ) {
        $taxonomy = get_taxonomy( $_taxonomy );
        if( $taxonomy->show_ui ) {
            $taxonomies[]['taxonomy_repeater_name'] = $taxonomy->labels->singular_name;
        }
    }
    return $taxonomies;
}
endif;

/**
 * Order options used for product query
 *
 * @since 3.0
 * @author al imran akash <alimranakash@gmail.com>
 *
 * @return pages
 */
if( !function_exists( 'wcd_get_pages' ) ) :
function wcd_get_pages() {

    $_pages = get_pages();

    $pages  = [ '0' => __( '- Choose a page -', 'codesigner-pro' ) ];
    foreach( $_pages as $page ) :
        $pages[ $page->ID ] = $page->post_title;
    endforeach;

    return apply_filters( 'wcd_get_pages', $pages );
}
endif;

/**
 * Order options used for product query
 *
 * @since 1.0
 *
 * @return []
 */
if( !function_exists( 'wcd_product_options' ) ) :
function wcd_product_options() {
    $options = [
        'name'              => __( 'Name', 'codesigner' ),
        'price'             => __( 'Price', 'codesigner' ),
        'description'       => __( 'Description', 'codesigner' ),
        'short_desc'        => __( 'Short Description', 'codesigner' ),
        'rating'            => __( 'Rating', 'codesigner' ),
        'review_count'      => __( 'Review Count', 'codesigner' ),
        'link'              => __( 'Link', 'codesigner' ),
    ];

    return apply_filters( 'codesigner-product_options', $options );
}
endif;

/**
 * Return the template types
 *
 * @since 3.0
 * @param $args, give array value. unset field
 * @author Al Imran Akash <alimranakash.bd@gmail.com>
 */
if( !function_exists( 'wcd_get_meta_fields' ) ) :
function wcd_get_meta_fields( $args = [] ) {
    global $wpdb;

    $all_ids = get_posts( array(
        'post_type'     => 'product',
        'numberposts'   => -1,
        'post_status'   => 'publish',
        'fields'        => 'ids',
    ) );

    $table_name = $wpdb->prefix . 'postmeta';

    $ids = implode( ',', $all_ids );
    
    $sql = "SELECT DISTINCT `meta_key` FROM `{$wpdb->prefix}postmeta` WHERE `post_id` IN( {$ids} )";
    

    $results = $wpdb->get_results( $sql );

    $meta_fields = [];
    foreach ( $results as $result ) {
        if ( !in_array( $result->meta_key, $args ) ) {
            $meta_fields[ $result->meta_key ] = ucwords( str_replace( '_', ' ', $result->meta_key ) );
        }
    }
    
    return $meta_fields;
}
endif;

/**
 * Return list of woocommerce email hooks
 *
 * @since 3.0
 * @author Jakaria Istauk<jakariamd35@gmail.com>
 */
if( !function_exists( 'wcd_get_wc_email_hooks' ) ) :
function wcd_get_wc_email_hooks() {
    
    $hooks = [
        'woocommerce_email_order_details'       => __( 'In order details', 'codesigner-pro' ),
        'woocommerce_email_before_order_table'  => __( 'Before order table', 'codesigner-pro' ),
        'woocommerce_email_after_order_table'   => __( 'After order table', 'codesigner-pro' ),
        'woocommerce_email_order_meta'          => __( 'In order meta', 'codesigner-pro' ),
        'woocommerce_email_customer_details'    => __( 'In customer details', 'codesigner-pro' ),
    ];
    
    return $hooks;
}
endif;

/**
 * Default checkout fields
 *
 * @param string $section form section billing|shipping|order
 *
 * @since 1.0
 */
if( !function_exists( 'wcd_wc_fields' ) ) :
function wcd_wc_fields( $section = '' ) {
    $fields = [
        'billing' => [ 'billing_first_name', 'billing_last_name', 'billing_company', 'billing_country', 'billing_address_1', 'billing_address_2', 'billing_city', 'billing_state', 'billing_postcode', 'billing_phone', 'billing_email' ],
        'shipping' => [ 'shipping_first_name', 'shipping_last_name', 'shipping_company', 'shipping_country', 'shipping_address_1', 'shipping_address_2', 'shipping_city', 'shipping_state', 'shipping_postcode' ],
        'order' => [ 'order_comments' ]
    ];

    if( $section != '' && isset( $fields[ $section ] ) ) {
        return apply_filters( 'wcd_wc_fields', $fields[ $section ] );
    }

    return apply_filters( 'wcd_wc_fields', $fields );
}
endif;

/**
 * Return list of woocommerce thankyou page hooks
 *
 * @since 3.0
 * @author Jakaria Istauk<jakariamd35@gmail.com>
 */
if( !function_exists( 'wcd_get_wc_thankyou_hooks' ) ) :
function wcd_get_wc_thankyou_hooks() {
    
    $hooks = [
        'woocommerce_before_thankyou'                   => __( 'Before Thankyou', 'codesigner-pro' ),
        'woocommerce_thankyou'                          => __( 'After/Bottom of Thankyou', 'codesigner-pro' ),
        'woocommerce_order_details_after_order_table'   => __( 'After Order Table', 'codesigner-pro' ),
    ];
    
    return $hooks;
}
endif;

/**
 * Return list of woocommerce order edit page hooks
 *
 * @since 3.0
 * @author Jakaria Istauk<jakariamd35@gmail.com>
 */
if( !function_exists( 'wcd_get_wc_edit_order_hooks' ) ) :
function wcd_get_wc_edit_order_hooks() {
    
    $hooks = [      
        'woocommerce_admin_order_data_after_order_details'      => __( 'After order details', 'codesigner-pro' ),
        'woocommerce_admin_order_data_after_billing_address'    => __( 'After billing address', 'codesigner-pro' ),
        'woocommerce_admin_order_data_after_shipping_address'   => __( 'After shipping address', 'codesigner-pro' ),
        'woocommerce_admin_order_item_headers'                  => __( 'Order item headers', 'codesigner-pro' ),
        'woocommerce_admin_order_totals_after_shipping'         => __( 'After shipping totals', 'codesigner-pro' ),
        'woocommerce_order_actions_start'                       => __( 'Before order action', 'codesigner-pro' ),
        'woocommerce_order_actions_end'                         => __( 'After order action', 'codesigner-pro' ),
    ];
    
    return $hooks;
}
endif;


/**
 * Determines if the pro version is activated
 *
 * @since 1.0
 */
if( !function_exists( 'wcd_is_pro_activated' ) ) :
function wcd_is_pro_activated() {
    return Helper::license()->_is_activated();
}
endif;

if( !function_exists( 'wcd_is_order_pay_page' ) ):
    function wcd_is_order_pay_page(){
        global $wp;
        $wcd_is_order_pay_page = ( !empty( $wp->query_vars['order-pay'] ) || isset( $_GET['pay_for_order'], $_GET['key'] ) );

        return $wcd_is_order_pay_page == 1 ? $wcd_is_order_pay_page : 0;
    }
endif;

if( !function_exists( 'wcd_woocommerce_checkout_payment' ) ):
    function wcd_woocommerce_checkout_payment() {
        echo wcd_pro_get_template( 'review-order-2', 'widgets/order-review/template-parts' );
    }
endif;

/**
 * Return list of woocommerce order edit page hooks
 *
 * @since 3.0
 * @author Jakaria Istauk<jakariamd35@gmail.com>
 */
if( !function_exists( 'wcd_get_current_order_id' ) ) :
function wcd_get_current_order_id() {

    global $wp_query;

    $post_id = get_the_ID();

    if ( get_post_type( $post_id ) == 'shop_order' ) {      
        return $post_id;        
    }


    elseif ( isset( $wp_query->query ) && isset( $wp_query->query['order-received'] ) ) {
        return $wp_query->query['order-received'];
    }

    /**
     * The last hope
     * @author Nazmul Ahsan
     */
    elseif( is_user_logged_in() ) {
        $customer   = new \WC_Customer( get_current_user_id() );
        $last_order = $customer->get_last_order();

        if ( is_a( $last_order, 'WC_Order' ) ) {
            return $last_order->get_id();
        }
        return false;
    }

    /**
     * Access order id from global variable
     * @author Ismail Hossen
     */
    else {
		global $cd_order_id;

        return $cd_order_id;
    }
    
    return false;
}
endif;

/**
 * Renders the output based on settings
 *
 * @since 1.0
 *
 * @return string
 */
if( !function_exists( 'wcd_render_message' ) ) :
function wcd_render_message( $settings, $echo = true ) {

    $message = '';

    if( !isset( $settings['show_message'] ) || $settings['show_message'] == 'text' ) {;

        $message = isset( $settings['message_text'] ) ? $settings['message_text'] : '';

    }

    if( !isset( $settings['show_message'] ) || $settings['show_message'] == 'template' ) {
        if ( isset( $settings['message_template'] ) && $settings['message_template'] != '' ) {
            $elementor_instance = \Elementor\Plugin::instance();
            $message = $elementor_instance->frontend->get_builder_content_for_display( $settings['message_template']  );
        }
    }

    if ( $echo ) {
        echo $message;
        return;
    }

    return $message;
}
endif;

/**
 * Get All orders IDs for a given product ID.
 *
 * @param  array  $product_ids (required)
 * @param  array    $order_status (optional) Default is 'wc-completed'
 * 
 * @since 3.0.0
 * 
 * @author Jakaria Istauk <jakariamd35@gmail.com>
 * @source https://stackoverflow.com/questions/45848249/woocommerce-get-all-orders-for-a-product
 * 
 * @return array
 */
function wcd_get_product_orders( $product_ids, $order_status = array( 'wc-completed' ), $user_id = 0 ){
    global $wpdb;

    $left_join = $sql_extention = '';

    if ( $user_id && $user_id != 0 ) {
        $user_id = codesigner_sanitize_number( $user_id );
        $left_join .= "LEFT JOIN {$wpdb->postmeta} AS meta ON order_items.order_id = meta.post_id";
        $sql_extention .= "AND meta.meta_key = '_customer_user' AND meta.meta_value = {$user_id}";

    }

    $sql = "
        SELECT order_items.order_id
        FROM {$wpdb->prefix}woocommerce_order_items as order_items
        LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta as order_item_meta ON order_items.order_item_id = order_item_meta.order_item_id
        LEFT JOIN {$wpdb->posts} AS posts ON order_items.order_id = posts.ID
        {$left_join}
        WHERE posts.post_type = 'shop_order' 
        {$sql_extention}
        AND posts.post_status IN ( '" . implode( "','", $order_status ) . "' )
        AND order_items.order_item_type = 'line_item'
        AND order_item_meta.meta_key = '_product_id'
        AND order_item_meta.meta_value IN ( '" . implode( "','", $product_ids ) . "' )
    ";

    $results = $wpdb->get_col( $sql );

    return $results;
}

/**
 * Renders the output based on settings
 *
 * @since 3.0
 * 
 * @author Jakaria Istauk <jakariamd35@gmail.com>
 *
 * @return string
 */
if( !function_exists( 'wcd_is_eligible' ) ) :
function wcd_is_eligible( $settings ) {

    $should_render  = false;
    $order_status   = function_exists( 'wc_get_order_statuses' ) ? wc_get_order_statuses() : [];
    $condition_list = $settings['codesigner_condition_list'];
    $current_user_id = get_current_user_id();
    $condition_count = 0;
    foreach ( $condition_list as $key => $condition_set ) {
        $this_condition = false;
        if ( $condition_set['condition_base'] == 'user' && $condition_set['is_user'] == 'loggedin' && is_user_logged_in() ) {
            $this_condition = true;
        }
        else if ( $condition_set['condition_base'] == 'user' && $condition_set['is_user'] == 'loggedout' && !is_user_logged_in() ) {
            $this_condition = true;
        }
        else if ( $condition_set['condition_base'] == 'user_id' && is_user_logged_in() && $condition_set['is_user_id'] == $current_user_id ) {
            $this_condition = true;
        }
        else if ( $condition_set['condition_base'] == 'user_role' && is_user_logged_in() ) {
            $user = wp_get_current_user();
            $roles = ( array ) $user->roles;
            if ( in_array( $condition_set['is_user_role'], $roles ) ) {
                $this_condition = true;
            }

        }
        else if ( $condition_set['condition_base'] == 'date_time' ) {
            $given_date = $condition_set['date'];
            $given_date = strtotime( $given_date );
            $relation   = $condition_set['date_condition'];
            $now        = time();

            $this_condition = ( $relation == 'greater_than' && $now > $given_date ) || ( $relation == 'less_than' && $now < $given_date );
        }
        else if ( $condition_set['condition_base'] == 'query_string' ) {
            global $wp_query;
            $query_string   = $wp_query->query_vars + $_GET;
            $query_key      = $condition_set['query_key'];
            $query_value    = $condition_set['query_value'];
            $relation       = $condition_set['query_relation'];
            
            $this_condition = ( $relation == 'equal' && array_key_exists( $query_key, $query_string ) && $query_string[ $query_key ] == $query_value ) || ( $relation == 'not_equal' && array_key_exists( $query_key, $query_string ) && $query_string[ $query_key ] != $query_value ) || ( $relation == 'greater_than' && array_key_exists( $query_key, $query_string ) && $query_string[ $query_key ] > $query_value ) || ( $relation == 'less_than' && array_key_exists( $query_key, $query_string ) && $query_string[ $query_key ] < $query_value );
        }
        else if ( $condition_set['condition_base'] == 'customer_purchased' ) {
            $product_ids = $condition_set['product_ids'];
            if ( $product_ids ) {
                $_product_ids = explode(',', $product_ids);
                $product_ids  = [];
                foreach ( $_product_ids as $key => $id ) {
                    $product_ids[] = codesigner_sanitize_number( $id );
                }
                $orders         = wcd_get_product_orders( $product_ids, array_keys( $order_status ), $current_user_id );
                $is_user_logged_in = is_user_logged_in();
                $this_condition = ( $condition_set['purchased'] == 'purchased' && !empty( $orders ) && $is_user_logged_in ) || ( $condition_set['purchased'] == 'not_purchased' && ( empty( $orders ) || !$is_user_logged_in ) );
            }
        }
        else if ( $condition_set['condition_base'] == 'customer_order' ) {
            $statuses   = $condition_set['order_status'];
            $orders     = wc_get_orders(array(
                'customer_id' => $current_user_id,
                'return' => 'ids',
                'status' => $statuses,
            ) );

            if( !empty( $orders ) ) $this_condition = true;
        }

        if( $condition_count == 0 && $this_condition ){
            $should_render = true;
        }

        if ( $condition_count > 0 ) {
            if ( $condition_set['conditions_relation'] == 'and' ) {
                $should_render = $should_render && $this_condition;
            }
            else{
                $should_render = $should_render || $this_condition;
            }
        }
        $condition_count++;
    }

    return $should_render;
}
endif;

/**
 * Print category hierarchical for only advance filter
 *@link https://wordpress.stackexchange.com/questions/14652/how-to-show-a-hierarchical-terms-list
 * 
 */
if ( ! function_exists( 'get_terms_hierarchical' ) ) {

    function get_terms_hierarchical( $terms, $parent_id = 0, $level = 0 , $_taxonomy = '', $type = '', $show_count = '' ) {

        foreach ( $terms as $term ) {

            if ( $term->name == 'Uncategorized' ) continue;

            if ( $parent_id == $term->parent ) {
                $checked    = '';
                $count_html = '';
                if ( $show_count == 'yes' ) {
                    $count_html = " ($term->count)";
                }
                $color      = get_term_meta( $term->term_id, 'wl-color' ) ? get_term_meta( $term->term_id, 'wl-color', true ) : '#000000';
                if( isset( $_GET['filter']['taxonomies'][$_taxonomy] ) && in_array( $term->slug, $_GET['filter']['taxonomies'][$_taxonomy] ) ) $checked = 'checked';
                ?>
                <div>
                    <?php echo str_repeat( '&nbsp;&nbsp;&nbsp;', $level ); ?>
                    <input id="<?php esc_attr_e( $_taxonomy ).'_'. esc_attr_e( $term->slug ); ?>" 
                    class="wl-fv-<?php esc_attr_e( $type ); ?>-custom" 
                    name="filter[taxonomies][<?php esc_attr_e( $_taxonomy ); ?>][]" 
                    type="<?php esc_attr_e( $type ); ?>" value="<?php esc_attr_e( $term->slug ); ?>" <?php esc_attr_e( $checked ); ?> >
                    <label for="<?php esc_attr_e( $_taxonomy ).'_'. esc_attr_e( $term->slug ); ?>" class="wl-fv-<?php esc_attr_e( $type ); ?>-custom-label" >
                        <p class="swtch-color" style="background-color:<?php esc_attr_e( $color ); ?>"></p>
                        <span><?php echo esc_html( $term->name.$count_html ); ?></span>
                    </label>
                </div>
                <?php
                get_terms_hierarchical( $terms, $term->term_id, $level + 1, $_taxonomy, $type, $show_count );
            }
        }
    }

}

if ( ! function_exists( 'get_user_password_reset_url' ) ) {
    function get_user_password_reset_url( $user_login ){  
           
        $is_email       = filter_var( $user_login, FILTER_VALIDATE_EMAIL );
        if ( $is_email ) {
            $user	= get_user_by( 'email', $is_email );
            $key	= get_password_reset_key( $user );
            $url	= esc_url( add_query_arg( array( 'key' => $key, 'id' => $user->ID ), wc_get_endpoint_url( 'lost-password', '', wc_get_page_permalink( 'myaccount' ))));  
            return $url ;          
        }
        else{
            $user	= get_user_by( 'login', $user_login );
            $key 	= get_password_reset_key( $user );
            $url	= esc_url( add_query_arg( array( 'key' => $key, 'id' => $user->ID ), wc_get_endpoint_url( 'lost-password', '', wc_get_page_permalink( 'myaccount' ))));
            return $url ; 
        }
    }
}